﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web.Mvc;
using BMS.Facade;
using BMS.Facade.Data;
using BMS.Facade.Fault;
using BMS.Utils;
using BMS.Web.App_GlobalResource;
using BMS.Web.Models;
using InfoWorld.HL7.ITS;
using Microsoft.Web.Mvc;

namespace BMS.Web.Controllers
{
    /// <summary>
    /// Ward Occupancy Controller class.
    /// </summary>
    [ValidateInput(false)]
    public class WardOccupancyController : BaseController
    {
        /// <summary>
        /// Default action for the Ward Occupancy Controller.
        /// </summary>
        /// <param name="extensionDivisionId">The extension division id.</param>
        /// <param name="rootDivisionId">The root division id.</param>
        /// <param name="divisionName">Name of the division.</param>
        /// <param name="bedStatusCode">The bed status code.</param>
        /// <param name="isFromSummaryReport">if set to <c>true</c> [is from summary report].</param>
        /// <param name="summaryReportReturnPath">The summary report return path (either facility or visn report).</param>
        /// <param name="summaryReportReturnText">The summary report return text (either facility or visn report).</param>
        /// <param name="visnRoot">The visn root used by visn bed summary report.</param>
        /// <param name="visnExtension">The visn extension used by visn bed summary report.</param>
        /// <returns>
        /// Returns the Ward Occupancy view result.
        /// </returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Index(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                WardOccupancyViewModel wardOccupancyModel = new WardOccupancyViewModel();
                if (string.IsNullOrWhiteSpace(QueryStrings["divisionId"]))
                    return this.RedirectToAction<HomeController>(act => act.Index());
                Division division = null;
                if (QueryStrings["divisionId"].Equals(Guid.Empty.ToString()))
                    division = new Division() { Name = Strings.All, Id = new II(loggedUser.Domain, QueryStrings["divisionId"]) };
                else
                    division = FacadeManager.EntityInterface.GetDivisionById(new II(loggedUser.Domain, QueryStrings["divisionId"]));
                wardOccupancyModel.Division = division.Name;
                wardOccupancyModel.ExtensionDivisionId = division.Id.extension;
                wardOccupancyModel.RootDivisionId = division.Id.root;
                wardOccupancyModel.BedStatusCode = QueryStrings["bedStatus"];
                wardOccupancyModel.IsFromSummaryReport = bool.Parse(QueryStrings["isFromSummaryReport"]);
                wardOccupancyModel.SummaryReportReturnPath = QueryStrings["summaryReportReturnPath"];
                wardOccupancyModel.SummaryReportReturnText = QueryStrings["summaryReportReturnText"];
                wardOccupancyModel.VisnRoot = loggedUser.Domain;
                wardOccupancyModel.VisnExtension = QueryStrings["visnId"];
                if (wardOccupancyModel.IsFromSummaryReport)
                {
                    Facility facility = FacadeManager.EntityInterface.GetFacilityByDivisionId(division.Id);
                    if (facility.Id.extension != this.loggedUser.Facility.Id.extension)
                    {
                        this.loggedUser.Facility = facility;
                        this.loggedUser.VistaSite = facility.VistaSite;
                        this.loggedUser.Visn = facility.VistaSite.Visn;
                        this.loggedUser.Region = facility.VistaSite.Visn.Region;
                        ViewBag.FacilityName = facility.Name;
                        ViewBag.FacilityCode = facility.Code;
                    }
                    if (string.IsNullOrEmpty(wardOccupancyModel.VisnExtension))
                        wardOccupancyModel.BedSummaryReportUrl = CreateBedSummaryReportUrl(wardOccupancyModel.SummaryReportReturnPath, wardOccupancyModel.SummaryReportReturnText);
                    else
                        wardOccupancyModel.BedSummaryReportUrl = CreateVISNBedSummaryReportUrl(wardOccupancyModel.SummaryReportReturnPath, wardOccupancyModel.SummaryReportReturnText, wardOccupancyModel.VisnRoot, wardOccupancyModel.VisnExtension);
                }

                FacilitySettings facilitySettings = FacadeManager.ConfigurationInterface.GetFacilitySettings(this.loggedUser.Facility.Id);

                DateTime lastDate = TimeZoneInfo.ConvertTime(DateTime.UtcNow, FacadeUtil.GetFacilityTimeZoneInfo(this.loggedUser.Facility.Id));
                wardOccupancyModel.LastUpdatedDate = string.Format("{0} {1} {2}", lastDate.ToString(Strings.USDate), Strings.At, lastDate.ToString("HH:mm"));
                if (facilitySettings != null && facilitySettings.FacilitySiteTimeZone != null)
                    wardOccupancyModel.FacilitySiteCodeTimeZone = string.Format("({0})", facilitySettings.FacilitySiteTimeZone.code);
                else
                    wardOccupancyModel.FacilitySiteCodeTimeZone = string.Empty;                

                List<Ward> wards = new List<Ward>();
                if (division != null && division.Id != null && !division.Id.extension.Equals((default(Guid)).ToString()))
                    wards = FacadeManager.EntityInterface.GetWardsByDivisionId(division.Id).ToList();
                else
                    wards = FacadeManager.EntityInterface.GetWardsByFacility(this.loggedUser.Facility.Id).ToList();

                var beds = FacadeManager.BedInterface.GetBedOccupancy(wards, division.Id, this.loggedUser.Facility, new CD { code = wardOccupancyModel.BedStatusCode }, false, null);
                wardOccupancyModel.IconFlagsList = FacadeManager.IconInterface.GetAllIconsByFacility(Guid.Parse(this.loggedUser.Facility.Id.extension)).Where(a => a.Active == true).ToList();
                wardOccupancyModel.BedInfoList = WardOccupancyBedViewModel.GetViewModel(wardOccupancyModel, beds);

                StringBuilder wardOccupancyReportUrl = new StringBuilder();                
                wardOccupancyReportUrl.Append("divisionId=");
                wardOccupancyReportUrl.Append(division.Id.extension);
                wardOccupancyReportUrl.Append("&bedStatusCode=");
                wardOccupancyReportUrl.Append(wardOccupancyModel.BedStatusCode);
                wardOccupancyModel.WardOccupancyReportUrl = "/Reporting/LocalReportViewer.aspx?" + CustomEncryption.Encrypt(wardOccupancyReportUrl.ToString(), loggedUser.Salt) + "*wo";

                //set total LOS admin and total LOS ward
                wardOccupancyModel.TotalLosAdmin = GetTotalLos(beds.Select(a => a.LosAdmin.Days).Sum(), beds.Select(a => a.LosAdmin.Hours).Sum());
                wardOccupancyModel.TotalLOSWard = GetTotalLos(beds.Select(a => a.LosWard.Days).Sum(), beds.Select(a => a.LosWard.Hours).Sum());

                wardOccupancyModel.LoggedUser = loggedUser;
                return View(wardOccupancyModel);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the total los.
        /// </summary>
        /// <param name="days">The days.</param>
        /// <param name="hours">The hours.</param>
        /// <returns></returns>
        private string GetTotalLos(int days, int hours)
        {
            if (hours > 24)
            {
                int addDays = Convert.ToInt32(hours / 24);
                days = days + addDays;
                hours = hours - (addDays * 24);
            }
            //format total LOS  
            return string.Format("{0}{1}{2}", days < 10 ? string.Format("{0}{1}", 0, days) : days.ToString(), ":", hours < 10 ? string.Format("{0}{1}", 0, hours) : hours.ToString());

        }

        #region Reports URL

        /// <summary>
        /// Creates URL for the VISN bed summary report.
        /// </summary>
        /// <param name="returnPath">The return path.</param>
        /// <param name="returnText">The return text.</param>
        /// <param name="visnRoot">The visn root.</param>
        /// <param name="visnExtension">The visn extension.</param>
        /// <returns>URL of VISN bed summary report.</returns>
        private string CreateVISNBedSummaryReportUrl(string returnPath, string returnText, string visnRoot, string visnExtension)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                II visnId = new II { root = visnRoot, extension = visnExtension };
                string visnNumber = FacadeManager.EntityInterface.GetVisn(visnId).Number;
                IList<Facility> facilites = FacadeManager.EntityInterface.GetFacilitiesByVisn(visnId).Where(a => a.IsAlive == true).ToList();
                string parameters = string.Empty;
                if (facilites != null)
                {
                    parameters = FacadeManager.ConfigurationInterface.GetFacMedicalCenterIds(this.loggedUser.Domain, facilites.Select(a => a.Id).ToList());
                }

                IReportInfo report = FacadeManager.ReportsInterface.GetReports(FullUserName).Where(a => a.Name == Constants.VISNBedSummaryReport).FirstOrDefault();
                if (report != null)
                {
                    //create paramets for report
                    if (report.Parameters == null)
                        report.Parameters = new Dictionary<String, String>();

                    report.Parameters.Add(Constants.REP_GENERAL_RETURN_TEXT, string.IsNullOrEmpty(returnText) ? Strings.ReturnToBedBoardHomePage : returnText);
                    report.Parameters.Add(Constants.REPORT_TITLE, string.Format("{0} {1} {2}", Strings.VISN.ToUpper(), visnNumber, Strings.BedSummaryReportTitle));
                    report.Parameters.Add(Constants.REP_GENERAL_RETURN_PATH, string.IsNullOrEmpty(returnPath) ? string.Format("{0}://{1}", this.Request.Url.Scheme, this.Request.Headers["host"]) : returnPath);
                    report.Parameters.Add(Constants.REP_GENERAL_FACILITIES, parameters);
                    report.Parameters.Add(Constants.REP_VISN_BED_SUMMARY_VISN_ROOT, visnRoot);
                    report.Parameters.Add(Constants.REP_VISN_BED_SUMMARY_VISN_EXTENSION, visnExtension);
                    report.Parameters.Add(Constants.REP_USER_SALT, loggedUser.Salt);

                    return report.Url;
                }
                else
                    throw new ReportRenderException(string.Format("{0} {1} {2}", Strings.VISN.ToUpper(), visnNumber, Strings.BedSummaryReportTitle), Strings.ERR_ReportNotAvailable);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates URL of facility bed summary report.
        /// </summary>
        /// <param name="returnPath">The return path.</param>
        /// <param name="returnText">The return text.</param>
        /// <returns>URL of Facility Bed Occupancy Summary Report.</returns>
        private string CreateBedSummaryReportUrl(string returnPath, string returnText)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IReportInfo report = Facade.FacadeManager.ReportsInterface.GetReports(FullUserName).Where(a => a.Name == Constants.BED_SUMMARY_REPORT).FirstOrDefault();

                if (report == null)
                {
                    throw new ReportRenderException(Constants.BED_SUMMARY_REPORT, Strings.ERR_ReportNotAvailable);
                }
                else
                {
                    if (report.Parameters == null)
                        report.Parameters = new Dictionary<String, String>();

                    report.Parameters.Add(Constants.REP_GENERAL_RETURN_PATH, string.IsNullOrEmpty(returnPath) ? string.Format("{0}://{1}", this.Request.Url.Scheme, this.Request.Headers["host"]) : returnPath);
                    report.Parameters.Add(Constants.REP_GENERAL_FACILITY_EXTENSION, this.loggedUser.Facility.Id.extension);
                    report.Parameters.Add(Constants.REP_TIME_ZONE_MINUTES_OFFSET, RepTimeZoneMinutesOffset);
                    report.Parameters.Add(Constants.REP_GENERAL_RETURN_TEXT, string.IsNullOrEmpty(returnText) ? Strings.ReturnToBedBoardHomePage : returnText);
                    report.Parameters.Add(Constants.REPORT_TITLE, Strings.BedSummaryReportTitle + ": " + this.loggedUser.Facility.Name);
                    report.Parameters.Add(Constants.REP_USER_SALT, loggedUser.Salt);
                }

                return report.Url;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion
    }
}
